//package globalExamples;

import oracle.olapi.data.source.DataProvider;
import oracle.olapi.data.source.CustomModel;
import oracle.olapi.data.source.Model;
import oracle.olapi.data.source.NumberSource;
import oracle.olapi.data.source.Source;
import oracle.olapi.data.source.StringSource;

import oracle.olapi.metadata.mdm.MdmMeasure;
import oracle.olapi.metadata.mdm.MdmPrimaryDimension;
import oracle.olapi.metadata.mdm.MdmLevelHierarchy;

/**
 * Complete code for Example 6-11, Implementing the extract Method as a
 * CustomModel, in Chapter 6, Understanding Source Objects, in the Oracle
 * OLAP Developer's Guide to the OLAP API.
 *
 * This program uses the Context10g class, which uses the
 * CursorPrintWriter class.
 *
 * @author Oracle Corporation
 */
public class ImplementingExtractAsACustomModel
{

  public void run(String[] args)
  {
    // Create a PrintWriter and objects for connecting to the database and
    // getting the metadata.
    Context10g context = new Context10g(args, false);

    context.println("Example 6-11, Implementing the extract Method as a CustomModel\n");

    // Get the DataProvider.
    DataProvider dp = context.getDataProvider();

    // Get the MdmMeasure objects and the Source objects for them.
    MdmMeasure mdmUnitPrice = context.getMdmMeasureByName("UNIT_PRICE");
    MdmMeasure mdmUnitCost = context.getMdmMeasureByName("UNIT_COST");

    NumberSource unitPrice = (NumberSource) mdmUnitPrice.getSource();
    NumberSource unitCost = (NumberSource) mdmUnitCost.getSource();

    // Get the MdmPrimaryDimension objects and the default hierarchies for them.
    MdmPrimaryDimension mdmProdDim = context.getMdmPrimaryDimensionByName("PRODUCT");
    MdmPrimaryDimension mdmTimeDim = context.getMdmPrimaryDimensionByName("TIME");

    MdmLevelHierarchy mdmProdRollup = (MdmLevelHierarchy)
                                         mdmProdDim.getDefaultHierarchy();
    MdmLevelHierarchy mdmCalendar = (MdmLevelHierarchy)
                                         mdmTimeDim.getDefaultHierarchy();

    // Get the Source objects for the hierarchies.
    StringSource prodRollup = (StringSource) mdmProdRollup.getSource();
    StringSource calendar = (StringSource) mdmCalendar.getSource();

    // Create a Source that represents a selection of Product dimension members.
    Source prodSel = prodRollup.selectValues(new String[]
                                             {"PRODUCT_ROLLUP::ITEM::13",
                                              "PRODUCT_ROLLUP::ITEM::14",
                                              "PRODUCT_ROLLUP::ITEM::15"});

    // Create a Source that is the result of a calculation involving two measures.
    Source calculation = unitPrice.minus(unitCost);

    // Create a Source whose element values are the Source objects for the measures and the
    // calculation. The resulting Source is like a measure dimension.
    Source sourceListSrc = dp.createListSource(new Source[] {unitPrice, unitCost, calculation});

    // Use the extract method and then join Source objects that match the inputs of the
    // Source objects in the list.
    Source resultUsingExtract = sourceListSrc.extract()
                                             .join(sourceListSrc)
                                             .join(prodSel)
                                             .join(calendar, "CALENDAR::MONTH::47");

    // Prepare and commit the current Transaction.
    context.commit();

    // Create a Cursor for the query and display the values of the Cursor.
    context.println("The values using extract method are:");
    context.displayResult(resultUsingExtract);

    // Produce the same result using a CustomModel directly.
    CustomModel  customModel = dp.createModel(sourceListSrc);
    customModel.assign(unitPrice.getID(), unitPrice);
    customModel.assign(unitCost.getID(), unitCost);
    customModel.assign(calculation.getID(), calculation);

    Source measValForSrc = customModel.createSolvedSource();

    // Join Source objects that match the inputs of the solved Source produced by
    // the CustomModel.
    Source resultUsingCustomModel = measValForSrc.join(sourceListSrc)
                                                 .join(prodSel)
                                                 .join(calendar, "CALENDAR::MONTH::47");

    context.commit();
    context.println("The values of using a CustomModel directly, with the Source IDs "
                     + "\nas the qualifications are:");
    context.displayResult(resultUsingCustomModel);

    // Create a list Source that has String objects as its element values.
    Source stringListSrc = dp.createListSource(new String[] {"price", "cost", "markup"});

    // Create a CustomModel for the list Source.
    CustomModel  customModel2 = dp.createModel(stringListSrc);
    customModel2.assign("price", unitPrice);
    customModel2.assign("cost", unitCost);
    customModel2.assign("markup", calculation);


    Source measValForSrc2 = customModel2.createSolvedSource();

    Source resultUsingCustomModel2 = measValForSrc2.join(stringListSrc)
                                                   .join(prodSel)
                                                   .join(calendar, "CALENDAR::MONTH::47");

    context.commit();
    context.println("The values using a CustomModel without using the Source IDs  " +
                     "\nas the qualifications are:");
    context.displayResult(resultUsingCustomModel2);

  }

  public static void main(String[] args)
  {
    new ImplementingExtractAsACustomModel().run(args);
  }
}


